#include "llvm/Pass.h"
#include "llvm/IR/Function.h"
#include "llvm/Support/raw_ostream.h"
#include "llvm/IR/Instructions.h"
#include "llvm/IR/IRBuilder.h"
#include "llvm/IR/Operator.h"
#include "ConstantPropagationInstruction.h"

using namespace llvm;

namespace {
struct OurConstantPropagationPass : public FunctionPass {
  std::vector<Value *> Variables;
  std::vector<ConstantPropagationInstruction *> Instructions;

  static char ID;
  OurConstantPropagationPass() : FunctionPass(ID) {}

  void findAllInstructions(Function &F)
  {
    for (BasicBlock &BB : F) {
      for (Instruction &I : BB) {
        ConstantPropagationInstruction *CPI = new ConstantPropagationInstruction(&I, Variables);
        Instructions.push_back(CPI);

        Instruction *Previous = I.getPrevNonDebugInstruction();

        if (Previous == nullptr) {
          for (BasicBlock *Pred : predecessors(&BB)) {
            Instruction *Terminator = Pred->getTerminator();
            CPI->addPredecessor(*std::find_if(Instructions.begin(), Instructions.end(),
            [Terminator](ConstantPropagationInstruction *CPI){ return CPI->getInstruction() == Terminator; }));
          }
        }
        else {
          CPI->addPredecessor(Instructions[Instructions.size() - 2]);
        }
      }
    }
  }

  void findAllVariables(Function &F)
  {
    for (BasicBlock &BB : F) {
      for (Instruction &I : BB) {
        if (isa<AllocaInst>(&I)) {
          Variables.push_back(&I);
        }
      }
    }
  }

  void setStatusForFirstInstruction()
  {
    for (Value *Variable : Variables) {
      Instructions.front()->setStatusBefore(Variable, Top);
    }
  }
  
  bool runOnFunction(Function &F) override {
    findAllVariables(F);
    findAllInstructions(F);
    setStatusForFirstInstruction();
    return true;
  }
}; // end of struct Hello
}  // end of anonymous namespace

char OurConstantPropagationPass::ID = 0;
static RegisterPass<OurConstantPropagationPass> X("our-constant-propagation", "Our simple constant propagation pass",
                             false /* Only looks at CFG */,
                             false /* Analysis Pass */);