/* Dat je jednostavan jezik aritmetickih izraza koji omogucava definisanje celobrojnih 
 * promenljivih i njihov prikaz. Napisati LL(1) parser koji proverava sintaksnu ispravnost
 * programa napisanih na datom jeziku. 
 * 
 * Primer:
 * a = 2+3*(4/b + 6 - 2*(4*c - 3));
 * print(-2*a);
 */

/*
Naivna gramatika:
program ->  program naredba     (levo rekurzivno pravilo)
            | naredba
naredba->   id = E ;
            | print (E);
E ->    E + T                   (levo rekurzivno pravilo)
        | T
T ->    T * F                   (levo rekurzivno pravilo)
        | F
F ->    broj
        | id
        | ( E )
        | - broj                (unarni minus)

Eliminisemo levu rekruziju:

program: naredba programP       
programP: naredba programP      
    | eps                       
naredba: ID = E ;               
         PRINT_T ( E ) ;        
E   -> T EP                     
EP  -> + T EP                   
       - T EP                   
       eps                      
T   -> F TP                     
TP  -> * F TP                   
       / F TP                   
       eps                      
F   -> ( E )                    
       broj                     
       ID                       
       - broj

Odredjujemo skupove izbora:

Simbol      |   First           | Eps   |   Follow
-------------------------------------------------------------
program     | id, print         | /     |   EOI
-------------------------------------------------------------
programP    | id, print         | Da    |   EOI
-------------------------------------------------------------
naredba     | id, print         | /     |   id, print
-------------------------------------------------------------
E           | id, broj, (, -    | /     |   ;, )
-------------------------------------------------------------
EP          | +, -              | Da    |   ;, )
-------------------------------------------------------------
T           | id, broj, (, -    | /     |   ;, )
-------------------------------------------------------------
TP          | *, /              | Da    |   +, -, ;, )
-------------------------------------------------------------
F           | id, broj, (, -    | /     |   *, /, +, -, ;, )
-------------------------------------------------------------


Konacna gramatika sa odredjenim skupovima izbora:

program: naredba programP       ID, PRINT_T
programP: naredba programP      ID, PRINT_T
    | eps                       EOI 
naredba: ID = E ;               ID
         PRINT_T ( E ) ;        PRINT_T
E   -> T EP                     BROJ, (, ID, -
EP  -> + T EP                   +
       - T EP                   -
       eps                      ;, )
T   -> F TP                     BROJ, (, ID, -
TP  -> * F TP                   *
       / F TP                   /
       eps                      +. -. ;, )
F   -> ( E )                    (
       broj                     broj
       ID                       ID
       - broj                   -

Dobijena gramatika je LL(1), tj. u svakom pravilu je moguce na osnovu jednog
preduvidnog simbola zakljuciti koji je sledeci korak u izvodjenju. Gramatika je LL(1)
ako:
1. Nije levo rekurzivna, niti levo faktorisana. 
2. Skupovi izbora su disjunktni. Preciznije, ako pravilo ima vise od jedne moguce grane
   tada svi preseci skupova izbora po parovima moraju biti prazni. 

Sledi implementacija parsera pomocu rekurzivnog spusta. Rekurzivni spust koristi
sistemski stek da bi cuvao tekuce izvodjenje. Svaki neterminal predstavljen je 
funkcijom istog naziva, dok se samo parsiranje odvija rekurzivnim pozivima cime se 
simulira zamena leve strane pravila desnom stranom pravila. 
*/

#include <iostream>
#include <cstdlib>

/* ukljucujemo spisak tokena */
#include "tokeni.h"

/* definisemo svoj mehanizam debagovanja, tj. pracenja izvodjenja */
#define DEBUG (1)

/* linkujemo leksicki analizator */
extern int yylex();

/* kreiramo promenljivu u kojoj cemo pamtiti preduvidni simbol, tj. sledeci token na ulazu */
int preduvid = 0;

/* deklaracije funkcija koje predstavljaju neterminale */
void program(void);
void programP(void);
void naredba(void);
void E(void);
void EP(void);
void T(void);
void TP(void);
void F(void);

/* glavni program */
int main() {

    /* ucitavamo prvi token sa ulaza*/
    preduvid = yylex();

    /* sintaksnu analizu pokrecemo od startnog simbola */
    program();

    /* ako je sve u redu, na kraju cemo kao sledeci token na ulazu imati EOI */
    if (preduvid == EOI)
        std::cout << "Sve je ok" << std::endl;
    else 
        std::cout << "Greska" << std::endl;

    exit(EXIT_SUCCESS);
}

/* Implementacija samog parsera 
 * sve funkcije se implementiraju na isti nacin. u zavisnosti od vrednosti tokena na ulazu
 * pravimo odgovarajuce rekurzivne pozive ili upadamo u obradu gresaka 
 * 
 * tokene mozemo samo da proveravamo i ponistavamo, tako sto ucitamo naredbi token.
 */

/* neterminal: 
 * program -> naredba programP      (id, print)
 */
void program(void) {

    /* ako se na ulazu nadje nesto iz skupa izbora datog pravila */
    if (preduvid == ID_T || preduvid == PRINT_T) {
#ifdef DEBUG
    /* u koraku debagovanja prikazujemo izvodjenje koje smo izvrsili */
    std::cout << "program -> naredba programP" << std::endl;
#endif  
        /* rekurzivno pozivamo odgovarajuce funkcije, tj. neterminale */
        naredba();
        programP();
        /* ovaj korak odgovara zamenjivanju leve strane pravila (program), 
         * desno stranom pravila (naredba programP)
         */
    }
    /* ako na ulazu nije nesto sto pripada mogucim skupovima izbora */
    else {
        /* prijavljujemo odgovarajucu gresku i prekidamo parsiranje */
        std::cerr << "Ocekivao sam: ID_T ili PRINT_T" 
            << std::endl;
        exit(EXIT_FAILURE);
    }
}

/* neterminal: 
 * programP -> naredba programP     (id, print)
 *             | eps                (EOI)
 * 
 * Neterminal ima vise grana sa razlicitim skupovima izbora, pa moramo sve da ih proverimo 
 */
void programP(void) {

    /* ako se na ulazu nadje nesto iz prvog skupa izbora datog pravila */
    if (preduvid == ID_T || preduvid == PRINT_T) {
#ifdef DEBUG
    /* u koraku debagovanja prikazujemo izvodjenje koje smo izvrsili */
    std::cout << "programP -> naredba programP" << std::endl;
#endif
        /* rekurzivno pozivamo odgovarajuce funkcije, tj. neterminale */
        naredba();
        programP();
        /* ovaj korak odgovara zamenjivanju leve strane pravila (programP), 
         * desno stranom pravila (naredba programP)
         */
    }
    /* ako se na ulazu nadje nesto iz drugog skupa izbora datog pravila */
    else if (preduvid == EOI) {
#ifdef DEBUG
    /* u koraku debagovanja prikazujemo izvodjenje koje smo izvrsili */
    std::cout << "programP -> eps" << std::endl;
#endif

        /* pravilo je epsilon, tj. anulirajuce, pa njemo odgovara prekidanje rekurzije */
        return;
    }
    /* ako na ulazu nije nesto sto pripada mogucim skupovima izbora */
    else {
        /* prijavljujemo odgovarajucu gresku i prekidamo parsiranje */
        std::cerr << "Ocekivao sam: ID_T ili PRINT_T ili EOI" 
            << std::endl;
        exit(EXIT_FAILURE);
    }
}

/* neterminal: 
 * naredba -> id = E ;              (id)
 *          | print ( E ) ;         (print)
 * 
 * Neterminal ima vise grana sa razlicitim skupovima izbora, pa moramo sve da ih proverimo 
 */
void naredba(void) {

    /* ako se na ulazu nadje nesto iz prvog skupa izbora datog pravila */
    if (preduvid == ID_T) {
#ifdef DEBUG
    /* u koraku debagovanja prikazujemo izvodjenje koje smo izvrsili */
    std::cout << "naredba -> ID = izraz ;" << std::endl;
#endif
        /* id smo vec iskoristili, pa ucitavamo sledeci token */
        preduvid = yylex();
        /* proveravamo da li je sledeci token onaj koji ocekujemo, tj. jednako */
        if (preduvid != JEDNAKO_T) {
            /* prijavljujemo odgovarajucu gresku i prekidamo parsiranje */
            std::cerr << "Ocekivao sam: JEDNAKO_T" 
                << std::endl;
            exit(EXIT_FAILURE);
        }
        /* ako jeste, ponisimo ga, tj. ucitamo sledeci token */
        preduvid = yylex();
        /* proverimo da li je izraz ispravan */
        E();
        /* ako je izraz ispravan, kao sledeci token ocekujemo ; */
        if (preduvid != TZ_T) {
            /* prijavljujemo odgovarajucu gresku i prekidamo parsiranje */
            std::cerr << "Ocekivao sam: TZ_T" 
                << std::endl;
            exit(EXIT_FAILURE);
        }
        /* ako je token bio ;, ponistavamo ga, tako sto ucitavamo sledeci*/
        preduvid = yylex();
    }
    /* ako se na ulazu nadje nesto iz drugog skupa izbora datog pravila */
    else if (preduvid == PRINT_T) {
#ifdef DEBUG
    /* u koraku debagovanja prikazujemo izvodjenje koje smo izvrsili */
    std::cout << "naredba -> PRINT_T ( E ) ;" << std::endl;
#endif  
        /* print smo iskoristili pa ucitavamo naredbi token */
        preduvid = yylex();
        /* proveravamo da li je sledeci token onaj koji ocekujemo, tj. ( */
        if (preduvid != OZ_T) {
            /* prijavljujemo odgovarajucu gresku i prekidamo parsiranje */
            std::cerr << "Ocekivao sam: OZ_T" 
                << std::endl;
            exit(EXIT_FAILURE);
        }
        /* ako jeste, ucitavamo sledeci token */
        preduvid = yylex();
        /* proveravamo da li je izraz ispravan */
        E();
        /* ako jeste, ocekujemo ) kao sledeci token */
        if (preduvid != ZZ_T) {
            /* prijavljujemo odgovarajucu gresku i prekidamo parsiranje */
            std::cerr << "Ocekivao sam: ZZ_T" 
                << std::endl;
            exit(EXIT_FAILURE);
        }
        /* ako se radi o zagradi, ucitavamo sledeci token sa ulaza */
        preduvid = yylex();
        /* ako je token onaj koji ocekujemo, tj. ;  */
        if (preduvid != TZ_T) {
            /* prijavljujemo odgovarajucu gresku i prekidamo parsiranje */
            std::cerr << "Ocekivao sam: TZ_T" 
                << std::endl;
            exit(EXIT_FAILURE);
        }
        /* ponistavamo ga tako sto ucitamo naredni token */
        preduvid = yylex();
    }
    /* ako na ulazu nije nesto sto pripada mogucim skupovima izbora */
    else {
        /* prijavljujemo odgovarajucu gresku i prekidamo parsiranje */
        std::cerr << "Ocekivao sam: ID_T ili PRINT_T" 
                << std::endl;
            exit(EXIT_FAILURE);
    }
}
/* neterminal: 
 * E -> T EP      (broj, id, (, -)
 */
void E(void) {

    /* ako se na ulazu nadje nesto iz skupa izbora datog pravila */
    if (preduvid == BROJ_T || preduvid == ID_T 
            || preduvid == OZ_T || preduvid == MINUS_T) {
#ifdef DEBUG
    /* u koraku debagovanja prikazujemo izvodjenje koje smo izvrsili */
    std::cout << "E -> T EP" << std::endl;
#endif
        /* rekurzivno pozivamo odgovarajuce funkcije, tj. neterminale */
        T();
        EP();
    }
    /* ako na ulazu nije nesto sto pripada mogucim skupovima izbora */
    else {
        /* prijavljujemo odgovarajucu gresku i prekidamo parsiranje */
        std::cerr << "Ocekivao sam: BROJ_T, ID_T, OZ_T" 
                << std::endl;
            exit(EXIT_FAILURE);
    }
}
/* neterminal: 
 * EP -> + T EP         (+)
 *     | - T EP         (-)
 *     | eps            (;, ))
 * 
 * Neterminal ima vise grana sa razlicitim skupovima izbora, pa moramo sve da ih proverimo 
 */
void EP(void) {
    /* ako se radi o prvoj grani */
    if (preduvid == PLUS_T) {
#ifdef DEBUG
    /* u koraku debagovanja prikazujemo izvodjenje koje smo izvrsili */
    std::cout << "EP -> + T EP" << std::endl;
#endif
        /*token + smo vec iskoristili, pa ucitavamo naredbi */
        preduvid = yylex();
        /* rekurzivno pozivamo odgovarajuce funkcije, tj. neterminale */
        T();
        EP();
    }
    /* ako se radi o drugoj grani pravila */
    else if (preduvid == MINUS_T) {
#ifdef DEBUG
    /* u koraku debagovanja prikazujemo izvodjenje koje smo izvrsili */
    std::cout << "EP -> - T EP" << std::endl;
#endif
        /*token - smo vec iskoristili, pa ucitavamo naredbi */
        preduvid = yylex();
        /* rekurzivno pozivamo odgovarajuce funkcije, tj. neterminale */
        T();
        EP();
    }
    /* ako se radi o anulirajucoj grani */
    else if (preduvid == ZZ_T || preduvid == TZ_T) {
#ifdef DEBUG
    /* u koraku debagovanja prikazujemo izvodjenje koje smo izvrsili */
    std::cout << "EP -> eps" << std::endl;
#endif  
        /* anuliramo pravilo, tako sto zaustavimo rekurziju */
        return;
    }
    /* ako na ulazu nije nesto sto pripada mogucim skupovima izbora */
    else {
        /* prijavljujemo odgovarajucu gresku i prekidamo parsiranje */
        std::cerr << "Ocekivao sam: +,-,),;" 
                << std::endl;
            exit(EXIT_FAILURE);
    }
}

/* neterminal: 
 * T -> F TP      (broj, id, (, -)
 */
void T(void) {

    /* ako se na ulazu nadje nesto iz skupa izbora datog pravila */
    if (preduvid == BROJ_T || preduvid == OZ_T 
            || preduvid == ID_T || preduvid == MINUS_T) {
#ifdef DEBUG
    /* u koraku debagovanja prikazujemo izvodjenje koje smo izvrsili */
    std::cout << "T -> F TP" << std::endl;
#endif      
        /* rekurzivno pozivamo odgovarajuce funkcije, tj. neterminale */
        F();
        TP();
    }
    /* ako na ulazu nije nesto sto pripada mogucim skupovima izbora */
    else {
        /* prijavljujemo odgovarajucu gresku i prekidamo parsiranje */
        std::cerr << "Ocekivao sam: BROJ_T, ID_T, (" 
                << std::endl;
            exit(EXIT_FAILURE);
    }
}
/* neterminal: 
 * TP -> + F TP         (*)
 *     | - F TP         (/)
 *     | eps            (+, -, ;, ))
 * 
 * Neterminal ima vise grana sa razlicitim skupovima izbora, pa moramo sve da ih proverimo 
 */
void TP(void) {

    /* ako se radi o prvoj grani pravila */
    if (preduvid == PUTA_T) {
#ifdef DEBUG    
    /* u koraku debagovanja prikazujemo izvodjenje koje smo izvrsili */
    std::cout << "TP -> * F TP" << std::endl;
#endif  
        /*token * smo vec iskoristili, pa ucitavamo naredbi */
        preduvid = yylex();
        /* rekurzivno pozivamo odgovarajuce funkcije, tj. neterminale */
        F();
        TP();
    }
    /* ako se radi o drugoj grani pravila */
    else if (preduvid == PODELJENO_T) {
#ifdef DEBUG
    /* u koraku debagovanja prikazujemo izvodjenje koje smo izvrsili */
    std::cout << "TP -> / F TP" << std::endl;
#endif
        /*token / smo vec iskoristili, pa ucitavamo naredbi */
        preduvid = yylex();
        /* rekurzivno pozivamo odgovarajuce funkcije, tj. neterminale */
        F();
        TP();
    }
    /* ako se radi o anulirajucoj grani */
    else if (preduvid == PLUS_T || preduvid == MINUS_T 
        || preduvid == TZ_T || preduvid == ZZ_T) {
#ifdef DEBUG
    /* u koraku debagovanja prikazujemo izvodjenje koje smo izvrsili */
    std::cout << "TP -> eps" << std::endl;
#endif
        /* anuliramo pravilo, tako sto zaustavimo rekurziju */
        return;
    }
    /* ako na ulazu nije nesto sto pripada mogucim skupovima izbora */
    else {
        /* prijavljujemo odgovarajucu gresku i prekidamo parsiranje */
        std::cerr << "Ocekivao sam: +,-,*,/,;,)" 
                << std::endl;
            exit(EXIT_FAILURE);
    }
}
/* neterminal: 
 * F -> - broj        (-)
 *    | broj          (broj)
 *    | id            (id)
 *    | ( E )         (()
 * 
 * Neterminal ima vise grana sa razlicitim skupovima izbora, pa moramo sve da ih proverimo 
 */
void F(void) {
    /* ako se radi o prvoj grani */
    if (preduvid == MINUS_T) {
#ifdef DEBUG
    /* u koraku debagovanja prikazujemo izvodjenje koje smo izvrsili */
    std::cout << "F -> - broj" << std::endl;
#endif        
        /* minus smo vec potrosili, pa ucitavamo sledeci simbol */
        preduvid = yylex();
        /* proveravamo da li je ocekivani token na ulazu, tj. broj*/
        if (preduvid != BROJ_T) {
            /* prijavljujemo odgovarajucu gresku i prekidamo parsiranje */
            std::cerr << "Ocekivao sam: BROJ_T" 
                << std::endl;
            exit(EXIT_FAILURE);
        }
        /* ako jeste, ponistavamo ga tako sto ucitavamo sledeci token */
        preduvid = yylex();
    }
    /* ako se radi o drugoj grani */
    else if (preduvid == BROJ_T) {
#ifdef DEBUG
    /* u koraku debagovanja prikazujemo izvodjenje koje smo izvrsili */
    std::cout << "F -> broj" << std::endl;
#endif
        /* broj smo vec iskoristili, pa ucitavamo sledeci token */
        preduvid = yylex();
    }
    /* ako se radi o trecoj grani */
    else if (preduvid == ID_T) {
#ifdef DEBUG
    /* u koraku debagovanja prikazujemo izvodjenje koje smo izvrsili */
    std::cout << "F -> ID" << std::endl;
#endif
        /* id smo vec iskoristili, pa ucitavamo sledeci token */
        preduvid = yylex();
    }
    /* ako se radi o cetvrtoj grani */
    else if (preduvid == OZ_T) {
#ifdef DEBUG
    /* u koraku debagovanja prikazujemo izvodjenje koje smo izvrsili */
    std::cout << "F -> ( E )" << std::endl;
#endif  
        /* otvorenu zagradu smo vec iskoristili, pa ucitavamo sledeci token */
        preduvid = yylex();
        /* proveravamo da li je izraz sintaksno ispravan */
        E();
        /* ako jeste, proveravamo da li je na kraju ocekivani token, tj. zatvorena zagrada */
        if (preduvid != ZZ_T) {
            /* prijavljujemo odgovarajucu gresku i prekidamo parsiranje */
            std::cerr << "Ocekivao sam: ZZ_T" 
                << std::endl;
            exit(EXIT_FAILURE);
        }
        /* ako jeste, ponistavamo je ucitavanjem sledeceg tokena */
        preduvid = yylex();
    }
    /* ako na ulazu nije nesto sto pripada mogucim skupovima izbora */
    else {
        /* prijavljujemo odgovarajucu gresku i prekidamo parsiranje */
        std::cerr << "Ocekivao sam: BROJ_T, ID_T, OZ_T" 
                << std::endl;
            exit(EXIT_FAILURE);
    }
}