#include "Razlomak.h"

/* pomocni metod koji euklidovim algoritmom odredjuje
 * nzd dva cela broja
 */
int Razlomak::nzd(int x, int y) {

	/* euklidov algoritam se moze primeniti samo na prirodne brojeve */
	x = abs(x);
	y = abs(y);

	do {
		int r = x % y;
		x = y;
		y = r;
	} while (y != 0);

	return x;
}

/* Podrazumevani konstruktor (bez argumenata)
 * atributi se mogu inicijalizovati listom vrednosti
 * ili eksplicitno u telu konstruktora
 */
Razlomak::Razlomak() : _brojilac(0), _imenilac(1) {
	
	// _imenilac = 1;
	// _brojilac = 0;

	/* poruke stampamo samo da bismo pratili kada se objekti instanciraju */
	std::cout << "Kreiram objekat" << std::endl;
}
/* Parametrizovani konstruktor sa jednim argumentom
 * kojim se razlomak inicijalizuje kao ceo broj n/1
 */
Razlomak::Razlomak(int br) : _brojilac(br), _imenilac(1) {

	// _brojilac = br;
	// _imenilac = 1;

	/* poruke stampamo samo da bismo pratili kada se objekti instanciraju */
	std::cout << "Kreiram objekat" << std::endl;
}

/* Parametrizovani konstruktor sa dva argumenta
 * kojim se razlomak inicijalizuje kao ceo broj m/n
 */
Razlomak::Razlomak(int br, int im) {

	/* atribute inicijalizujemo u telu konstruktora */
	_brojilac = br;
	_imenilac = im;

	/* imenilac mora biti prirodan broj (>=1) */
	if (_imenilac < 0) {
		_brojilac = -_brojilac;
		_imenilac = -_imenilac;
	}

	/* ako je nula, izbacujemo izuzetak */
	if (_imenilac == 0)
		throw "imenilac 0";

	/* odredjujemo nzd vrednosti */
	int d = nzd(_brojilac, _imenilac);

	/* dovodimo razlomak u neskrativi oblik */
	_brojilac /= d;
	_imenilac /= d;

	/* poruke stampamo samo da bismo pratili kada se objekti instanciraju */
	std::cout << "Kreiram objekat" << std::endl;
}
/* Konstruktor kopije:
 * Implementiramo ga za vezbu. Nasa klasa je jednostavna i ne koristi nikakav
 * dodatni resurs, pa mozemo da se oslonima i na podrazumevano obezbedjeni konstruktor
 * kopije, koji radi samo plitku kopiju atributa klase. Mi kao atribute imamo proste tipove
 * podataka, a kod njih je ne postoji duboko kopiranje. 
 */
Razlomak::Razlomak(const Razlomak& r) : _brojilac(r._brojilac), _imenilac(r._imenilac){

	//_brojilac = r._brojilac;
	//_imenilac = r._imenilac;

	/* poruke stampamo samo da bismo pratili kada se objekti instanciraju */
	std::cout << "Kopiram objekat" << std::endl;
}

/* Desktruktor:
 * U okviru klase ne koristimo nijedan dinamicki resurs, pa formalno nema potrebe
 * da implementiramo eksplicitni destruktor. Ovde je implementiran samo da bismo
 * pratili opseg zivota objekta
 */
Razlomak::~Razlomak() {

	/* poruke stampamo samo da bismo pratili kada se objekat unistava, tj.
	 * da bismo ilustrovali opseg zivota objekta
	 */
	std::cout << "Unistavam objekat: " << (*this) << std::endl;
}
/* Operator dodele: (copy assignment)
 * Implementiramo ga samo za vezbu. Klasa Razlomak je jednostavna i mozemo da se oslonimo na 
 * default implementirani operator dodele, koji se svodi na poziv default konstruktora kopije 
 * i plitko kopiranje atributa. Medjutim, upravo jednostavnost ove klase jasno ilustruje razliku 
 * izmedju konstruktora kopije i operatora dodele.
 * 
 * BITNO:
 * Konstruktor kopije kreira novi objekat na osnovu postojeceg.
 * Operator dodele samo menja vrednosti atributa postojeceg objekta.
 * 
 * Ova razlika je kljucna za pravilno dizajniranje klasa i upravljanje resursima u C++.
 * Poistovecivanje konstruktora kopije sa operatorom dodele je potencijalni uzrok ozbiljnih 
 * bagova u programima. 
 * 
 * Ukratko:
 * 1. Konstruktor kopije se koristi kada objekat prvo mora da se kreira, pre nego sto kopiranje vrednosti
 *    moze da se desi. 
 *    (npr. prenosenje objekata funkcijama po vrednosti ili vracanje objekata po vrednosti iz funkcija)
 * 2. Operator dodele (copy assignment) se koristi u slucajevima kada objekat ne mora da se kreira pre
 *    nego sto kopiranje moze da se izvrsi. 
 * 
 * BITNO 2:
 * Primetite povratni tip operatora dodele. 
 * Radi se o referenci, dakle drugom imenu za postojeci objekat, sto je saglasno sa semantikom
 * operatora, prema kojoj samo menjamo vrednosti postojeceg objekta.
 */
Razlomak& Razlomak::operator =(const Razlomak& r) {

	/* stampamo poruku da bismo videli kada se desava poziv operatora dodele */
	std::cout << "Dodeljujem vrednost: " << r << " objektu " << (*this) << std::endl;

	/* BITNO: (self-assignment check)
	 * otkrivanje dodeljivanja samom sebi (self-assignment)
	 * Klasa Razlomak je vrlo jednostavna, pa mozda nije uocljivo odmah zasto je ovo neophodno.
	 * Za sada je dovoljno da vam je jasno da ne postoji potreba da objekat prepisuje svoj
	 * sadrzaj samim sobom. 
	 * Sa slozenijim klasama ovaj korak je neophodna i kljucna provera pri dodeli.
	 */
	if (this == &r)
		return *this;

	/* vrednosti argumenta dodeljuemo atributima klase nad kojim je pozvan argument*/
	_brojilac = r._brojilac;
	_imenilac = r._imenilac;

	/* Kao rezultat vracamo referencu na postojeci objekat, tj. *this.
	 * 
	 * BITNO:
	 * U programskom jeziku C++ promenljiva this oznacava pokazivac na samog sebe, a ne referencu.
	 * Dereferenciranjem dobijamo sam objekat (tj. *this je sam objekat, pa kao povratna vrednost ce
	 * biti samo novo ime za vec postojeci objekat u ovom slucaju)
	 */
	return *this;
}

/* get metod za brojilac
 * const u deklaraciji metoda oznacava da metod nece menjati 
 * atribute klase
 */
int Razlomak::Brojilac() const {

	return _brojilac;
}
/* set metod za brojilac 
 */
void Razlomak::Brojilac(int br) {

	/* pamtimo brojilac */
	_brojilac = br;

	/* svodimo razlomak na neskrativ */
	int d = nzd(_brojilac, _imenilac);

	_brojilac /= d;
	_imenilac /= d;

}

/* get metod za imenilac
 * const u deklaraciji metoda oznacava da metod nece menjati
 * atribute klase
 */
int Razlomak::Imenilac() const {

	return _imenilac;
}

/* set metod za imenilac
 */
void Razlomak::Imenilac(int im) {

	/* imenilac ne moze biti 0 */
	if (im == 0)
		throw "imenilac 0";

	/* pamtimo imenilac */
	_imenilac = im;
	/* vrsimo provere */
	if (_imenilac < 0) {
		_brojilac = -_brojilac;
		_imenilac = -_imenilac;
	}

	/* dovodimo razlomak na neskrativ oblik */
	int d = nzd(_brojilac, _imenilac);

	_brojilac /= d;
	_imenilac /= d;
}

/* metod prikazuje objekat na izlaznom strimu */
void Razlomak::show(std::ostream& s) const {

	s << _brojilac << "/" << _imenilac;
}

/* metod ucitava razlomak sa ulaznog strima */
void Razlomak::read(std::istream& s) {

	/* pomocne promenljive */
	char c;
	int br, im;
	/* ucitamo vrednosti */
	s >> br >> c >> im;

	/* izvrsimo provere */
	if (c != '/')
		throw "Neispravan format";

	if (im == 0)
		throw "imenilac 0";

	/* inicijalizujemo vrednosti */
	_brojilac = br;
	_imenilac = im;

	/* proverimo znake */
	if (_imenilac < 0) {
		_brojilac = -_brojilac;
		_imenilac = -_imenilac;
	}

	/* dovedimo razlomak u neskrativ oblik */
	int d = nzd(_brojilac, _imenilac);

	_brojilac /= d;
	_imenilac /= d;
}

/* Preopterecujemo operator za sabiranje 
 * 
 * BITNO:
 * Ako u kodu imate sledecu naredbu
 * p + q;
 * 
 * Ona se prevodi u sledeci oblik: (na objekat p primeni operator + sa argumentom q)
 * p.operator+(q);
 * 
 * ciji je rezultat novi objekat (medjurezultat) koji sadrzi
 * zbir dva razlomka.
 * 
 * BITNO 2:
 * 1. Pazljivo razmotrite deklaraciju operatora za sabiranje u svetlu prethodnog razlaganja
 *    poziva p+q. Treb da primetite da je objekat p implicitni objekat, dok se objekat q 
 *    eksplicitno prosledjuje kao argument operatora. 
 * 2. Primetite da se objekat q kao argument funkcije prenosi kao referenca, tj. samo kao drugo ime
 *    za postojeci objekat. Na ovaj nacin izbegava se nepotrebno kopiranje i ubrzava se izvrsavanje
 *    programa.
 * 3. Argument funkcije je obelezen kao const, sto znaci da se njegovo stanje (vrednosti atributa)
 *    nece menjati tokom izvrsavanja metoda.
 * 4. Sam operator je obelezen kao const na kraju, sto znaci da se nece menjati vrednosti implicitnog
 *    argumenta.
 * 
 * BITNO 3:
 * Upravo zbog ovakvog raspakivanja operatora morate pazljivo da vodite racuna o tome sta je 
 * prvi, a sta drugi argument operatora.
 */
Razlomak Razlomak::operator +(const Razlomak& r) const {

	/* racunamo zbir dva razlomka */
	int br = _brojilac * r._imenilac + r._brojilac * _imenilac;
	int im = _imenilac * r._imenilac;

	/* kao rezultat vracamo novi Razlomak */
	return Razlomak(br, im);
}

/* BITNO
 * Cesta greska prilikom preopterecivanja operatora je pokusaj da se bude pametan i da se izbegne
 * kopiranje prilikom vracanja rezultata. Takva greska se oslikava u sledecoj implementaciji:
 * 
 * Razlomak& Razlomak::operator +(const Razlomak& r) const {
 *		
 *		int br = _brojilac * r._imenilac + r._brojilac * _imenilac;
 *		int im = _imenilac * r._imenilac;
 *		Razlomak r(br, im);
 * 
 *		return r;
 * }
 * 
 * u kojoj se kao povratna vrednost navodi referenca. Imajuci u vidu raniju pricu o tome sta referenca
 * predstavlja, treba da vam bude jasno zasto je ovo budalastina. Referenca je samo drugo ime za
 * postojeci objekat. Problem se oslikava u tome sto se vracanjem reference na objekat koji je lokalna 
 * promenljiva, dobija referenca na nesto sto prestaje da postoji onog momenta kada se metod izvrsi.
 * Preciznije, kao povratna vrednost je vraceno nesto sto ne postoji i to je segfault istog trenutka. 
 * 
 * Dakle, iz funkcije kao rezultat treba da vratite novi objekat kao vrednost ili pokazivac na objekat
 * instanciran na hipu, jer samo tako vracate nesto sto zaista postoji i moze da se koristi u nastavku
 * programa. Vracanje objekata alociranih na hipu podrazumeva da ih oslobodite onog trenutka kada
 * vam vise ne trebaju.
 */

/* ista prica kao i + */
Razlomak Razlomak::operator -(const Razlomak& r) const {

	int br = _brojilac * r._imenilac - r._brojilac * _imenilac;
	int im = _imenilac * r._imenilac;

	return Razlomak(br, im);
}

/* ista prica kao i - */
Razlomak Razlomak::operator *(const Razlomak& r) const {

	int br = _brojilac * r._brojilac;
	int im = _imenilac * r._imenilac;

	return Razlomak(br, im);
}

/* ista prica kao i / */
Razlomak Razlomak::operator /(const Razlomak& r) const {

	int br = _brojilac * r._imenilac;
	int im = _imenilac * r._brojilac;

	return Razlomak(br, im);
}

/* unarni minus, tj. suprotni razlomak
 * std::cout << -p << std::endl;
 * 
 * Ovaj poziv se raspakuje u:
 * p.operator-();
 * 
 * i kao rezultat daje novi razlomak
 */
Razlomak Razlomak::operator -() const {
	
	return Razlomak(-_brojilac, _imenilac);
}

/* reciprocni razlomak, ista prica kao i unarni minus*/
Razlomak Razlomak::operator ~() const {

	if (_brojilac == 0)
		return Razlomak(_brojilac, _imenilac);

	return Razlomak(_imenilac, _brojilac);
}

/* Operatori inkrementiranja(++) i dekrementiranja(--)
 * 
 * Da biste mogli pravilno da razumete implementaciju
 * operatora inkrementiranja i dekrementiranja morate prvo da imate 
 * dobro razumevanje njihove semantike.
 * 
 * Semantika operatora je sledeca:
 * 1. Prefiksni operator - prvo se uvecava vrednost promenljive, pa se zatim koristi u izrazu
 * 2. Postfiksni operator - u izrazu se koristi stara vrednost promenljive, pa se onda uvecava 
 *                          njena vrednost.
 * 
 * Semantika operatora povlaci sledece zakljucke:
 * 1. Prefiksni - s obzirom da prvo uvecavam vrednost, onda mogu da izmenim stanje svog objekta
 *                i da direktno njega koristim u izrazu, sto samo znaci da ne moram da kreiram novi
 *				  objekat, vec mogu postojeci da vratim preko reference nakon izmene.
 * 2. Postfiksni - s obzirom da treba da koristim staru vrednost promenljive i da istovremeno promenim
 *				  stanje objekta, jasno je da to ne mogu da ucinim pomocu samo jednog objekta, jer nije
 *				  moguce da moj objekat istovremeno ima i staru i novu vrednost. Zbog toga treba prvo da
 *				  napravim kopiju trenutnog objekta (sacuvam startu vrednost), da uvecam/umanjim vrednost
 *				  trenutnog objekta i da kao rezultat vratim kopiju (preko vrednosti). Na taj nacin smo
 *				  osigurali pravilno izvrsavanje postfiksnih operatora.
 * 
 * Prilikom implementacije operatora, njihovo razlikovanje je vrlo lako:
 * 1. Prefiksni operatori nemaju argumente.
 * 2. Postfiksni operatori imaju jedan lazni int argument.
 */

/* Prefiksni operatori */

/* primetimo da kao rezultat vracamo referencu na sam objekat */
Razlomak& Razlomak::operator ++() {
	/* promenimo vrednost objekta */
	_brojilac += _imenilac;
	/* vratimo referencu na objekat */
	return *this;

	/* semantika - prvo promenim, pa koristim novu vrednost */
}
/* primetimo da kao rezultat vracamo referencu na sam objekat */
Razlomak& Razlomak::operator --() {
	/* promenimo vrednost objekta */
	_brojilac -= _imenilac;
	/* vratimo referencu na objekat */
	return *this;

	/* semantika - prvo promenim, pa koristim novu vrednost */
}

/* Postfiksni operatori */

/* primetimo da kao rezultat vracamo novi objekat, ne postojeci */
Razlomak Razlomak::operator++(int) {

	/* prvo zapamtimo staru vrednost trenutnog objekta */
	Razlomak r(*this);
	/* izmenitmo vrednost starog objekta */
	_brojilac += _imenilac;
	/* kao rezultat vratimo staru vrednost */
	return r;

	/* semantika - prvo koristim staru vrednost, pa izmenim stanje objekta */
}

/* primetimo da kao rezultat vracamo novi objekat, ne postojeci */
Razlomak Razlomak::operator--(int) {

	/* prvo zapamtimo staru vrednost trenutnog objekta */
	Razlomak r(*this);
	/* izmenitmo vrednost starog objekta */
	_brojilac -= _imenilac;
	/* kao rezultat vratimo staru vrednost */
	return r;
	/* semantika - prvo koristim staru vrednost, pa izmenim stanje objekta */
}

/* logicki operatori */
bool Razlomak::operator ==(const Razlomak& r) const {

	if (_brojilac == r._brojilac && _imenilac == r._imenilac)
		return true;

	return false;
}

bool Razlomak::operator !=(const Razlomak& r) const {

	return !(*this == r);
}

bool Razlomak::operator <=(const Razlomak& r) const {

	int br1 = _brojilac * r._imenilac;
	int br2 = _imenilac * r._brojilac;

	return br1 <= br2;
}

bool Razlomak::operator >=(const Razlomak& r) const {

	int br1 = _brojilac * r._imenilac;
	int br2 = _imenilac * r._brojilac;

	return br1 >= br2;
}

bool Razlomak::operator <(const Razlomak& r) const {

	int br1 = _brojilac * r._imenilac;
	int br2 = _imenilac * r._brojilac;

	return br1 < br2;
}

bool Razlomak::operator >(const Razlomak& r) const {

	int br1 = _brojilac * r._imenilac;
	int br2 = _imenilac * r._brojilac;

	return br1 > br2;
}

/* globalni operatori za pisanje i citanje razlomaka 
 * 
 * BITNO:
 * Operatori << i >> se uvek definisu kao globalni, nikada u okviru klase.
 * Kada biste ih definisali u okviru klase, morali biste da ih pozivate kontra-intuitivno, tj. na 
 * sledeci nacin
 * 
 * r << std::cout;
 * 
 * sto se prevodi u
 * r.operator<<(std::cout)
 * 
 * Jasno je da ovakav nacin stampanja nije ono sto zelimo, vec zelimo da stampanje bude kao:
 * std::cout << r << std::endl.
 * 
 * U prevodu, zelimo da naucimo operator << kako da pravilno ispise nesto tipa Razlomak u bilo kom
 * trenutku. To znaci da operator kao levi argument treba da ima neki postojeci ostream (tj. ostream&)
 * i kao desni argument neki postojeci Razlomak (tj. Razlomak&). Radi nadovezivanja stampanja, operator
 * kao povratnu vrednost treba da vrati isti onaj ostream na koji je pisao sadrzaj.
 * Isti slucaj je sa ulaznim operatorom >>.
 */

/* iz prethodnog sledi definicija operatora <<
 * BITNO:
 * 1. Argument ostream& se nikad ne obelezava kao const, jer se pri pisanju menja njegovo stanje
 * 2. Argument koji se ispisuje se uvek obelezava kao const, jer se prilikom pisanja ne sme 
 *    menjati njegovo stanje
 */
std::ostream& operator <<(std::ostream& s, const Razlomak& r) {
	
	/* objekat sam ume da se ispise na ostream */
	r.show(s);
	/* vratimo ostream radi nadovezivanja */
	return s;
}
/* iz prethodnog sledi definicija operatora >> 
 * BITNO:
 * 1. Argument istream& se nikad ne obelezava kao const, jer se pri citanju menja njegovo stanje
 * 2. Argument koji se ucitava se nikada ne obelezava kao const, jer se prilikom citanja menja
 *    njegovo stanje.
 */
std::istream& operator >>(std::istream& s, Razlomak& r) {
	
	/* objekat sam ume da se ucita sa istream */
	r.read(s);
	
	/* vratimo istream radi nadovezivanja */
	return s;
}