#include "Matrix.hpp"

/* konstrutkor pravi praznu matricu */
Matrix::Matrix() : _mat(){

}

/* konstruktor pravi matricu konstanti dimenzije mxn */
Matrix::Matrix(int m, int n, double val) :_mat(m, std::vector<double>(n, val)) {

}

/* konstruktor kopira matricu na koju pokazuje pokazivac m */
Matrix::Matrix(Matrix* m) : _mat() {

    /* prosirimo broj redova */
    _mat.resize(m->Rows());
    /* redom kopisamo element po element */
    for (int i = 0; i < m->Rows(); i++) {
        /* prosirimo svaki pojedinacni red */
        _mat[i].resize(m->Columns());
        /* prekopiramo elemente */
        for (int j = 0; j < m->Columns(); j++) {
            _mat[i][j] = (*m)[i][j];
        }
    }
}

/* metod dodaje red u matricu 
 * BITNO:
 * neophodno je proveriti da li je ovo dodavanje uopste moguce, tj.
 * da li red ima onoliko elemenata koliko je u matrici. ova provera
 * se moze izvesti na nivou klase, ali moze i u gramatici. prikazano resenje
 * ovu provere radi u akcijama gramatike.
 */
void Matrix::dodajRed(std::vector<double>& red) {

    _mat.push_back(red);
}

/* pomocni metod koji stampa matricu na ostream s */
void Matrix::show(std::ostream& s) const {

    for (int i = 0; i < _mat.size(); i++) {
        for (int j = 0; j < _mat[i].size(); j++) {
            /* setw postavlja sirinu stampe */
            std::cout << std::setw(6) << _mat[i][j] << " ";
        }
        s << std::endl;
    }
}

/* metod vraca broj redova matrice */
int Matrix::Rows() const {

    return _mat.size();
}

/* metod vraca broj kolona matrice */
int Matrix::Columns() const {

    /* pre vracanja broja kolona neophodno je proveriti da li
     * u matrici postoji makar jedan red. ako ne postoji nijedan red,
     * program bi pukao, jer bismo indeksirali prazan niz.
     */
    return _mat.size() > 0 ? _mat[0].size() : 0;
}

/* metod kao uredjeni par vraca dimenziju matrice 
 * druga ideja bi bila da dimenziju matrice vratimo kao matricu
 */
std::pair<int, int> Matrix::Size() const {

    /* prilikom kreiranja para zovemo metode Rows() i Columns 
     * koje vrse sve potrebne provere 
     */
    return std::make_pair(this->Rows(), this->Columns());
}

/* operator indeksiranja.
 * treba nam interno u klasi radi lakseg manipulisanja elementima matrice,
 * iako u gramatici ne postoji zahtev za indeksiranjem pojedinacnih elemenata.
 * 
 * BITNO:
 * nasa matrica je vektor vektora.
 * operator indeksiranja u C++ moze da indeksira samo jednu dimenziju zakljucno sa standardom C++20.
 * u nasem slucaju, operator indeksiranja ce indeksirati redove matrice, pa kao svoj rezultat
 * treba da vrati referencu na postojeci red (sto je vektor<double>). u kodu, drugi indeks ce zapravo
 * koristiti preoptereceni operator indeksiranja vektora i na taj nacin cemo simulirati indeksiranje
 * elemenata matrice.
 * 
 * programski jezik C++ ne podrzava preopterecivanje operatora indeksiranja tako da prihvataju 
 * vise argumenata zakljucno sa standardom C++20. dakle, nije moguce definisati sledeci
 * operator
 * double& Matrix::operator[](int i, int j) {
 *      return _mat[i][j];
 * }
 * 
 * vec se operator indeksiranja mora napisati kao sto sledi. 
 */
std::vector<double>& Matrix::operator[](int i) {

    /* provera gresaka */
    if (i < 0 || i >= _mat.size())
        throw "Index out of bounds";

    return _mat[i];
}

/* const verzija operatora */
const std::vector<double>& Matrix::operator[](int i) const {

    if (i < 0 || i >= _mat.size())
        throw "Index out of bounds";

    return _mat[i];
}

/* gramatika omogucava sabiranje sa skalarima.
 * skalare jednostavno mozemo da tumacimo kao matrice 1x1. 
 * u tom slucaju, samo uvecavamo svaki element drugog operanda
 * tim skalaram. Operacije sa skalarima se mogu implementirati
 * i kao odvojene funkcije, ako vam je tako preglednije 
 */
Matrix* Matrix::operator +(const Matrix& m) const {

    /* rezultat je na pocetku nullptr */
    Matrix* rezultat = nullptr;

    /* ako je levi operand skalar (1x1 matrica) */
    if (this->Rows() == 1 && this->Columns() == 1) {
        /* rezultat ima dimenzije desnog operanda */
        rezultat = new Matrix(m.Rows(), m.Columns());

        /* potrebno je samo da svaki element desnog operanda
         * uvecamo datim skalarom
         */
        for (int i = 0; i < m.Rows(); i++) {
            for (int j = 0; j < m.Columns(); j++) {
                rezultat->_mat[i][j] = m._mat[i][j] + _mat[0][0];
            }
        }
    }
    /* ako je desni operand skalar */
    else if (m.Rows() == 1 && m.Columns() == 1) {
        
        /* rezultat ima dimenzije levog operanda */
        rezultat = new Matrix(this->Rows(), this->Columns());

        /* potrebno je samo da svaki element levod operanda
         * uvecamo datim skalarom
         */
        for (int i = 0; i < this->Rows(); i++) {
            for (int j = 0; j < this->Columns(); j++) {
                rezultat->_mat[i][j] = _mat[i][j] + m._mat[0][0];
            }
        }
    }
    /* ako nijedan operand nije skalar, treba da izvrsimo klasicno
     * sabiranje matrica
     */
    else {
        /* ako matrice nisu istih dimenzija, 
         * vracamo nullptr kao rezultat.
         * 
         * BITNO:
         * S obzirom da poziv moze da ne uspe, potrebno je da
         * u akcijama svaki put proverimo povratnu vrednost nakon svake
         * operacije koja potencijalno prijavljuje gresku.
         */
        if (this->Size() != m.Size())
            return nullptr;
        
        /* racunamo zbir matrica */
        std::pair<int, int> size = this->Size();
        rezultat = new Matrix(size.first, size.second);

        for (int i = 0; i < size.first; i++) {
            for (int j = 0; j < size.second; j++) {
                (*rezultat)[i][j] = _mat[i][j] + m._mat[i][j];
            }
        }
    }

    /* vracamo rezultat */
    return rezultat;
}

/* operator oduzimanja. ista logika kao i operator + */
Matrix* Matrix::operator -(const Matrix& m) const {

    Matrix* rezultat = nullptr;

    if (this->Rows() == 1 && this->Columns() == 1) {
        rezultat = new Matrix(m.Rows(), m.Columns());

        for (int i = 0; i < m.Rows(); i++) {
            for (int j = 0; j < m.Columns(); j++) {
                rezultat->_mat[i][j] = m._mat[i][j] - _mat[0][0];
            }
        }
    }
    else if (m.Rows() == 1 && m.Columns() == 1) {
        
        rezultat = new Matrix(this->Rows(), this->Columns());

        for (int i = 0; i < this->Rows(); i++) {
            for (int j = 0; j < this->Columns(); j++) {
                rezultat->_mat[i][j] = _mat[i][j] - m._mat[0][0];
            }
        }
    }
    else {
        if (this->Size() != m.Size())
            return nullptr;
        
        std::pair<int, int> size = this->Size();
        rezultat = new Matrix(size.first, size.second);

        for (int i = 0; i < size.first; i++) {
            for (int j = 0; j < size.second; j++) {
                (*rezultat)[i][j] = _mat[i][j] + m._mat[i][j];
            }
        }
    }

    return rezultat;
}

/* operator mnozenja. ista logika kao kod operator + i - */
Matrix* Matrix::operator *(const Matrix& m) const {

    Matrix* rezultat = nullptr;

    if (this->Rows() == 1 && this->Columns() == 1) {
        rezultat = new Matrix(m.Rows(), m.Columns());

        for (int i = 0; i < m.Rows(); i++) {
            for (int j = 0; j < m.Columns(); j++) {
                rezultat->_mat[i][j] = m._mat[i][j] * _mat[0][0];
            }
        }
    }
    else if (m.Rows() == 1 && m.Columns() == 1) {
        
        rezultat = new Matrix(this->Rows(), this->Columns());

        for (int i = 0; i < this->Rows(); i++) {
            for (int j = 0; j < this->Columns(); j++) {
                rezultat->_mat[i][j] = _mat[i][j] * m._mat[0][0];
            }
        }
    }
    else {
        if (this->Columns() != this->Rows())
            return nullptr;

        rezultat = new Matrix(this->Rows(), m.Columns());

        for (int i = 0; i < this->Rows(); i++) {
            for (int j = 0; j < m.Columns(); j++) {
                rezultat->_mat[i][j] = 0;
                for (int k = 0; k < this->Columns(); k++) {
                    rezultat->_mat[i][j] += _mat[i][k] * m._mat[k][j];
                }
            }
        }
    }

    return rezultat;
}

/* unarni minus */
Matrix* Matrix::operator -() const {

    /* kreiramo matricu istih dimenzija kao originalna */
    Matrix* rezultat = new Matrix(this->Rows(), this->Columns());

    /* svaki element u novoj matrici je suprotni element polazne matrice */
    for (int i = 0; i < this->Rows(); i++) {
        for (int j = 0; j < this->Columns(); j++) {
            rezultat->_mat[i][j] = -_mat[i][j];
        }
    }

    /* vracamo rezultat */
    return rezultat;
}

Matrix* Matrix::Transponuj() const {

    Matrix* rezultat = new Matrix(this->Columns(), this->Rows());

    for (int i = 0; i < _mat.size(); i++) {
        for (int j = 0; j < _mat[0].size(); j++) {
            rezultat->_mat[j][i] = _mat[i][j];
        }
    }

    return rezultat;
}

/* pokoordinatno mnozenje matrica, ista logika kao i ostali operatori */
Matrix* Matrix::Pomnozi(const Matrix& m) const {

    Matrix* rezultat = nullptr;

    if (this->Rows() == 1 && this->Columns() == 1) {
        rezultat = new Matrix(m.Rows(), m.Columns());

        for (int i = 0; i < m.Rows(); i++) {
            for (int j = 0; j < m.Columns(); j++) {
                rezultat->_mat[i][j] = m._mat[i][j] * _mat[0][0];
            }
        }
    }
    else if (m.Rows() == 1 && m.Columns() == 1) {
        
        rezultat = new Matrix(this->Rows(), this->Columns());

        for (int i = 0; i < this->Rows(); i++) {
            for (int j = 0; j < this->Columns(); j++) {
                rezultat->_mat[i][j] = _mat[i][j] * m._mat[0][0];
            }
        }
    }
    else {
        if (this->Size() != m.Size())
            return nullptr;
        
        std::pair<int, int> size = this->Size();
        rezultat = new Matrix(size.first, size.second);

        for (int i = 0; i < size.first; i++) {
            for (int j = 0; j < size.second; j++) {
                (*rezultat)[i][j] = _mat[i][j] * m._mat[i][j];
            }
        }
    }

    return rezultat;
}

/* pokoordinatno deljenje matrica ista logika kao i ostali operatori  */
Matrix* Matrix::Podeli(const Matrix& m) const {

    Matrix* rezultat = nullptr;

    if (this->Rows() == 1 && this->Columns() == 1) {
        rezultat = new Matrix(m.Rows(), m.Columns());

        for (int i = 0; i < m.Rows(); i++) {
            for (int j = 0; j < m.Columns(); j++) {
                rezultat->_mat[i][j] = m._mat[i][j] * _mat[0][0];
            }
        }
    }
    else if (m.Rows() == 1 && m.Columns() == 1) {
        
        rezultat = new Matrix(this->Rows(), this->Columns());

        for (int i = 0; i < this->Rows(); i++) {
            for (int j = 0; j < this->Columns(); j++) {
                rezultat->_mat[i][j] = _mat[i][j] * m._mat[0][0];
            }
        }
    }
    else {
        if (this->Size() != m.Size())
            return nullptr;
        
        std::pair<int, int> size = this->Size();
        rezultat = new Matrix(size.first, size.second);

        for (int i = 0; i < size.first; i++) {
            for (int j = 0; j < size.second; j++) {
                (*rezultat)[i][j] = _mat[i][j] / m._mat[i][j];
            }
        }
    }

    return rezultat;
}

/* metod izdvaja podmatricu datih dimenzija */
Matrix* Matrix::SubMatrix(std::pair<int, int>& rows, std::pair<int, int>& cols) {

    /* rekonstrukcija dimenzija */
    int rowFrom = (rows.first == -1 ? 0 : rows.first);
    int rowTo = (rows.second == -1 ? (this->Rows() - 1) : rows.second);

    int colFrom = (cols.first == -1 ? 0 : cols.first);
    int colTo = (cols.second == -1 ? (this->Columns() - 1) : cols.second);

    /* provera da li su dimenzije ispravne */
    if ((rowTo < rowFrom) || (rowFrom < 0 || rowFrom >= Rows()) || (rowTo < 0 || rowTo >= Rows())
        || (colTo < colFrom) || (colFrom < 0 || colFrom >= Columns()) || (colFrom < 0 || colFrom >= Columns())) {
            return nullptr;
    }

    /* odredjivanje dimenzija*/
    int rows1 = rowTo - rowFrom + 1;
    int cols1 = colTo - colFrom + 1;

    /* kreiranje podmatrice */
    Matrix* rezultat = new Matrix(rows1, cols1);

    /* kopiranje elemenata */
    for (int i = rowFrom; i <= rowTo; i++) {
        for (int j = colFrom; j <= colTo; j++) {
            rezultat->_mat[i - rowFrom][j- colFrom] = _mat[i][j];
        }
    }

    /* vracanje rezultata */
    return rezultat;
}

/* uporedjivanje matrica */
bool Matrix::operator ==(const Matrix& m) const {

    if (this->Size() != m.Size())
        return false;

    for (int i = 0; i < Rows(); i++) {
        for (int j = 0; j < Columns(); j++) {
            if (_mat[i][j] != m[i][j])
                return false;
        }
    }

    return true;
}

/* uporedjivanje matrica */
bool Matrix::operator !=(const Matrix& m) const {
    
    return !(*this == m);
}

/* operator za stampanje */
std::ostream& operator <<(std::ostream& s, const Matrix& m) {

    m.show(s);
    return s;
}