#include "syn_tree.hpp"

ConstNode::ConstNode(int value) 
    : _value(value) {

    }
ConstNode::ConstNode(const ConstNode& cn)
    : _value(cn._value) {

    }

SynTreeNode* ConstNode::Clone() const {
    return new ConstNode(*this);
}
void ConstNode::Print(std::ostream& s) const {
    s << _value;
}
void* ConstNode::Interpret(SymbolTable& st) const {
    int* result = new int[1];
    
    result[0] = _value;

    return result;
}


StringNode::StringNode(const std::string& s) 
    : _id(s) {

}
StringNode::StringNode(const StringNode& id) 
    : _id(id._id){

}

SynTreeNode* StringNode::Clone() const {
    return new StringNode(*this);
}
void StringNode::Print(std::ostream& s) const {
    s << _id;
}
void* StringNode::Interpret(SymbolTable& st) const {

    return new std::string(_id);
}

GraneNode::GraneNode(SynTreeNode* left, SynTreeNode* right)
    : _left(left), _right(right) {

}
GraneNode::GraneNode(const GraneNode& gn) 
    : _left(nullptr), _right(nullptr){

    if (gn._left != nullptr)
        _left = gn._left->Clone();
    _right = gn._right->Clone();
}
GraneNode::~GraneNode() {
    if (_left != nullptr)
        delete _left;
    if (_right != nullptr)
        delete _right;
}

SynTreeNode* GraneNode::Clone() const {
    return new GraneNode(*this);
}
void GraneNode::Print(std::ostream& s) const {

    if (_left != nullptr) {
        _left->Print(s);
        s << ", ";
    }
    _right->Print(s);
}
void* GraneNode::Interpret(SymbolTable& st) const {
    std::vector<int> * result = nullptr;
    if (_left != nullptr) {
        result = (std::vector<int>*)(_left->Interpret(st));
    }
    else {
        result = new std::vector<int>();
    }

    int* val = (int*)(_right->Interpret(st));
    result->push_back(val[0]);
    delete[] val;

    return result;
}

CvorNode::CvorNode(SynTreeNode* left, SynTreeNode* right)
    : _left(left), _right(right) {

    }
CvorNode::CvorNode(const CvorNode& gn) 
    : _left(nullptr), _right(nullptr){

    if (gn._left != nullptr)
        _left = gn._left->Clone();
    if (gn._right != nullptr)
        _right = gn._right->Clone();
}
CvorNode::~CvorNode() {
    if (_left != nullptr)
        delete _left;
    if (_right != nullptr)
        delete _right;
}

SynTreeNode* CvorNode::Clone() const {
    return new CvorNode(*this);
}
void CvorNode::Print(std::ostream& s) const {

    _left->Print(s);
    s << " : [ ";
    if (_right != nullptr)
        _right->Print(s);
    s << " ]";
}
void* CvorNode::Interpret(SymbolTable& st) const {

    std::vector<int>* vec = nullptr;
    if (_right != nullptr)
        vec = (std::vector<int>*)_right->Interpret(st);
    else  
        vec = new std::vector<int>();
    
    int* value = (int*)_left->Interpret(st);
    int nodeNum = value[0];
    delete[] value;
    
    return new std::pair<int, std::vector<int>*>(nodeNum, vec);
}

ListaCvorovaNode::ListaCvorovaNode(SynTreeNode* left, SynTreeNode* right)
    : _left(left), _right(right) {

    }
ListaCvorovaNode::ListaCvorovaNode(const ListaCvorovaNode& gn)
    : _left(nullptr), _right(nullptr) {
    if (gn._left != nullptr)
        _left = gn._left->Clone();
    if (gn._right != nullptr)
        _right = gn._right->Clone();
    }
ListaCvorovaNode::~ListaCvorovaNode() {
    if (_left != nullptr)
        delete _left;
    if (_right != nullptr)
        delete _right;
}

SynTreeNode* ListaCvorovaNode::Clone() const {
    return new ListaCvorovaNode(*this);
}
void ListaCvorovaNode::Print(std::ostream& s) const {
    if (_left != nullptr) {

        _left->Print(s); 
        s << ", ";
    }
    _right->Print(s);
}
void* ListaCvorovaNode::Interpret(SymbolTable& st) const {
    Graph* res = nullptr;
    if (_left != nullptr) {

        res = (Graph*)_left->Interpret(st);
    }
    else {
        res = new Graph();
    }

    std::pair<int, std::vector<int>*>* par = (std::pair<int, std::vector<int>*>*)_right->Interpret(st);
    res->AddNode(par->first, par->second);
    delete par;

    return res;
}

GrafNode::GrafNode(SynTreeNode* exp)
    : _exp(exp) {

    }
GrafNode::GrafNode(const GrafNode& gn)
    : _exp(gn._exp->Clone()) {

    }
GrafNode::~GrafNode() {
    delete _exp;
}

SynTreeNode* GrafNode::Clone() const {
    return new GrafNode(*this);
}
void GrafNode::Print(std::ostream& s) const {
    s << "{";
    _exp->Print(s);
    s << "}";
}
void* GrafNode::Interpret(SymbolTable& st) const {
    Graph* res = (Graph*) _exp->Interpret(st);
    res->BuildMatrixRep();
    return res;;
}

IdNode::IdNode(const std::string& exp) 
    : _name(exp){

}
IdNode::IdNode(const IdNode& gn)
    : _name(gn._name){

}
IdNode::~IdNode() {

}

SynTreeNode* IdNode::Clone() const {
    return new IdNode(*this);
}
void IdNode::Print(std::ostream& s) const {
    s << _name;
}
void* IdNode::Interpret(SymbolTable& st) const {

    return st.Value(_name);
}

ZagradeNode::ZagradeNode(SynTreeNode* exp)
    : _exp(exp) {

}
ZagradeNode::ZagradeNode(const ZagradeNode& gn)
    : _exp(gn._exp->Clone()){

}
ZagradeNode::~ZagradeNode(){
    delete _exp;
}

SynTreeNode* ZagradeNode::Clone() const {

    return new ZagradeNode(*this);
}
void ZagradeNode::Print(std::ostream& s) const{
    s << "( "; 
    _exp->Print(s);
    s << " )";
}
void* ZagradeNode::Interpret(SymbolTable& st) const {

    return _exp->Interpret(st);
}

ComplNode::ComplNode(SynTreeNode* exp)
    : _exp(exp) {

}
ComplNode::ComplNode(const ComplNode& gn)
    : _exp(gn._exp->Clone()){

}
ComplNode::~ComplNode(){
    delete _exp;
}

SynTreeNode* ComplNode::Clone() const {
    return new ComplNode(*this);
}
void ComplNode::Print(std::ostream& s) const {
    s << "~";
    _exp->Print(s);
}
void* ComplNode::Interpret(SymbolTable& st) const {

    Graph* g = (Graph*)_exp->Interpret(st);
    Graph* res = ~(*g);
    delete g;

    return res;
}

BinaryOpNode::BinaryOpNode(SynTreeNode* left, SynTreeNode* right, const std::string& op)
    : _left(left), _right(right), _op(op){

}
BinaryOpNode::BinaryOpNode(const BinaryOpNode& gn)
    : _left(gn._left->Clone()), _right(gn._right->Clone()), _op(gn._op){

}
BinaryOpNode::~BinaryOpNode(){
    delete _left;
    delete _right;
}

SynTreeNode* BinaryOpNode::Clone() const {
    return new BinaryOpNode(*this);
}
void BinaryOpNode::Print(std::ostream& s) const {
    _left->Print(s);
    s << " " << _op << " ";
    _right->Print(s);
}
void* BinaryOpNode::Interpret(SymbolTable& st) const {

    Graph* levi = (Graph*)_left->Interpret(st);
    Graph* desni = (Graph*)_right->Interpret(st);

    Graph* res = nullptr;

    if (_op == "+")
        res = *levi + *desni;
    else if (_op == "*")
        res = *levi * *desni;

    delete levi;
    delete desni;

    return res;
}

PrintNode::PrintNode(SynTreeNode* exp, int type)
    : _exp(exp), _type(type){

}
PrintNode::PrintNode(const PrintNode& gn)
    : _exp(gn._exp->Clone()), _type(gn._type){

}
PrintNode::~PrintNode(){

    delete _exp;
}

SynTreeNode* PrintNode::Clone() const {

    return new PrintNode(*this);
}
void PrintNode::Print(std::ostream& s) const {

    switch (_type) {
        case 1:
            s << "print_matrix_d(";
            break;
        case 2:
            s << "print_matrix_u(";
            break;
        case 3:
            s << "print(";
            break;
    }
    _exp->Print(s);
    s << ");";
}
void* PrintNode::Interpret(SymbolTable& st) const {

    Graph* g = (Graph*)_exp->Interpret(st); 
    switch (_type) {
        case 1:
            std::cout << g->Directed() << std::endl;
            break;
        case 2:
            std::cout << g->Undirected() << std::endl;
            break;
        case 3:
            std::cout << (*g) << std::endl;
            break;
    }

    return nullptr;
}   

DodelaNode::DodelaNode(SynTreeNode* exp, const std::string& name)
    : _name(name), _exp(exp){

}
DodelaNode::DodelaNode(const DodelaNode& gn)
    : _name(gn._name), _exp(nullptr){

    if (gn._exp != nullptr)
        _exp = gn._exp->Clone();
}
DodelaNode::~DodelaNode(){

    if (_exp != nullptr)
        delete _exp;
}

SynTreeNode* DodelaNode::Clone() const{

    return new DodelaNode(*this);
}
void DodelaNode::Print(std::ostream& s) const{

    s << "graph " << _name;
    if (_exp != nullptr) {
        s << " = ";
        _exp->Print(s);
    }
    s << ";";
}
void* DodelaNode::Interpret(SymbolTable& st) const{

    Graph* g = nullptr;
    if (_exp != nullptr) {
        g = (Graph*)_exp->Interpret(st);
    }
    else {
        g = new Graph();
    }

    st.AddVariable(_name, g);
    return nullptr;
}

UpdateNode::UpdateNode(SynTreeNode* exp, const std::string& name)
    : _name(name), _exp(exp){

}
UpdateNode::UpdateNode(const UpdateNode& gn)
    : _name(gn._name), _exp(gn._exp->Clone()){

}
UpdateNode::~UpdateNode(){

        delete _exp;
}

SynTreeNode* UpdateNode::Clone() const{

    return new UpdateNode(*this);
}
void UpdateNode::Print(std::ostream& s) const{

    s << _name;
    s << " = ";
    _exp->Print(s);
    s << ";";
}
void* UpdateNode::Interpret(SymbolTable& st) const{

    Graph* g = (Graph*)_exp->Interpret(st);

    st.Update(_name, g);

    return nullptr;
}

ProgramNode::ProgramNode(SynTreeNode* exp)
    : _exp(exp) {

    }
ProgramNode::ProgramNode(const ProgramNode& gn)
    : _exp(gn._exp->Clone()) {

    }
ProgramNode::~ProgramNode() {
    delete _exp;
}

SynTreeNode* ProgramNode::Clone() const {
    return new ProgramNode(*this);
}
void ProgramNode::Print(std::ostream& s) const{

    s << "begin: " << std::endl;
    _exp->Print(s);
    s << std::endl << "end" << std::endl;
}
void* ProgramNode::Interpret(SymbolTable& st) const {

    return _exp->Interpret(st);
}

SequenceNode::SequenceNode()
    : _vec(){

}
SequenceNode::SequenceNode(const SequenceNode& gn)
    : _vec() {

    for (auto& it : gn._vec) {
        _vec.push_back(it->Clone());
    }
}
SequenceNode::~SequenceNode(){

    for (auto& it : _vec)
        delete it;
}

void SequenceNode::AddStatement(SynTreeNode* exp) {
    _vec.push_back(exp);
}

SynTreeNode* SequenceNode::Clone() const {
    return new SequenceNode(*this);
}
void SequenceNode::Print(std::ostream& s) const {

    for (auto& it : _vec) {
        it->Print(s);
        s << std::endl;
    }
}
void* SequenceNode::Interpret(SymbolTable& st) const {

    for (auto& it: _vec) {
        it->Interpret(st);
    }
    return nullptr;
}

DropNode::DropNode(const std::string& name, int node)
    : _name(name), _node(node) {

}

SynTreeNode* DropNode::Clone() const {
    return new DropNode(*this);
}
void DropNode::Print(std::ostream& s) const {
    s << "delete " << _name << " " << _node << ";";
}
void* DropNode::Interpret(SymbolTable& st) const {

    Graph* g = st.Value(_name);
    g->DropNode(_node);
    st.Update(_name, g);

    return nullptr;
}

AdjNode::AdjNode(const std::string& name, int node)
    : _name(name), _node(node) {

}

SynTreeNode* AdjNode::Clone() const {
    return new AdjNode(*this);
}
void AdjNode::Print(std::ostream& s) const {
    s << _name << "[" << _node << "];";
}
void* AdjNode::Interpret(SymbolTable& st) const {

    Graph* g = st.Value(_name);
    std::vector<int>* adjList = g->at(_node);
    std::cout << "[";
    for (int i = 0; i < adjList->size(); i++) {
        if (i > 0)
            std::cout << ", ";
        std::cout << adjList->at(i);
    }
    std::cout << "]" << std::endl;
    delete g;

    return nullptr;
}
