#include "syn_tree.hpp"

Skup univerzalni;

std::ostream& operator <<(std::ostream& s, const SynTreeNode* stn) {
    stn->show(s);
    return s;
}

BrojConstNode::BrojConstNode(int v)
    : _value(v) {

    }

Type* BrojConstNode::Execute(SymbolTable& st) const {

    return new IntType(_value);
}
void BrojConstNode::Compile(SymbolTable& st, std::ostream& s) const {
    s << _value;
}
SynTreeNode* BrojConstNode::Clone() const {
    return new BrojConstNode(*this);
}
void BrojConstNode::show(std::ostream& s) const {
    s << _value;
}

StringConstNode::StringConstNode(const std::string& s)
    : _value(s) {

    }

Type* StringConstNode::Execute(SymbolTable& st) const {

    return new StringType(_value);
}
void StringConstNode::Compile(SymbolTable& st, std::ostream& s) const {
    s << "\"" << _value << "\"";
}
SynTreeNode* StringConstNode::Clone() const {
    return new StringConstNode(*this);
}
void StringConstNode::show(std::ostream& s) const {
     s << "\"" << _value << "\"";
}

SizeNode::SizeNode(SynTreeNode* exp)
    : _exp(exp) {

    }
SizeNode::SizeNode(const SizeNode& sn)
    : _exp(sn._exp->Clone()) {

    }
SizeNode::~SizeNode() {
    delete _exp;
}

Type* SizeNode::Execute(SymbolTable& st) const {
    
    Type* result = _exp->Execute(st);
    SkupType* skup = dynamic_cast<SkupType*>(result);

    if (skup == nullptr)
        throw std::invalid_argument("Argument nije tipa skup");
    
    Type* size = new IntType(skup->GetSkup().Size());
    delete result;

    return size;
}
void SizeNode::Compile(SymbolTable& st, std::ostream& s) const {

    s << "size(";
    _exp->Compile(st, s);
    s << "";
}
SynTreeNode* SizeNode::Clone() const {
    return new SizeNode(*this);
}
void SizeNode::show(std::ostream& s) const {

    s << "size(";
    _exp->show(s);
    s << ")";
}

ListaBrojevaNode::ListaBrojevaNode(SynTreeNode* left, SynTreeNode* right)
    : _left(left), _right(right) {

    }
ListaBrojevaNode::ListaBrojevaNode(const ListaBrojevaNode& sn)
    : _left(nullptr), _right(nullptr) {

        if (sn._left != nullptr)
            _left = sn._left->Clone();
        if (sn._right != nullptr)
            _right = sn._right->Clone();
    }
ListaBrojevaNode::~ListaBrojevaNode() {

    if (_left != nullptr)
        delete _left;
    if (_right != nullptr)
        delete _right;
}

Type* ListaBrojevaNode::Execute(SymbolTable& st) const {
    Type* lresult = nullptr;
    if (_left != nullptr) 
        lresult = _left->Execute(st);

    Type* rresult = _right->Execute(st);
    IntType* broj = dynamic_cast<IntType*>(rresult);
    if (broj == nullptr)
        throw std::invalid_argument("Desni argument mora biti broj");
    
    if (lresult == nullptr) {
        SkupType* result = new SkupType(Skup(broj->GetInt()));
        delete broj;
        return result;
    }

    SkupType* skup = dynamic_cast<SkupType*>(lresult);
    if (skup == nullptr)
        throw std::invalid_argument("Levi argument mora biti broj");

    skup->Dodaj(broj->GetInt());
    delete broj;

    return skup;
}
void ListaBrojevaNode::Compile(SymbolTable& st, std::ostream& s) const{

    if (_left != nullptr) {
        _left->Compile(st, s);
        s << ", ";
    }
    _right->Compile(st, s);
}
SynTreeNode* ListaBrojevaNode::Clone() const{
    return new ListaBrojevaNode(*this);
}
void ListaBrojevaNode::show(std::ostream& s) const{

    if (_left != nullptr) {
        _left->show(s);
        s << ", ";
    }
    _right->show(s);
}

SkupNode::SkupNode(SynTreeNode* exp) 
    : _exp(exp) {

}
SkupNode::SkupNode(const SkupNode& sn) 
    : _exp(sn._exp->Clone()){

}
SkupNode::~SkupNode() {

    delete _exp;
}

Type* SkupNode::Execute(SymbolTable& st) const{

    return _exp->Execute(st);
}
void SkupNode::Compile(SymbolTable& st, std::ostream& s) const{

    s << "{";
    _exp->Compile(st, s);
    s << "}";
}
SynTreeNode* SkupNode::Clone() const{

    return new SkupNode(*this);
}
void SkupNode::show(std::ostream& s) const{

    s << "{";
    _exp->show(s);
    s << "}";
}

BinaryOperationNode::BinaryOperationNode(SynTreeNode* left, SynTreeNode* right, const std::string& op)
    : _left(left), _right(right), _op(op){

}
BinaryOperationNode::BinaryOperationNode(const BinaryOperationNode& b)
    : _left(b._left->Clone()), _right(b._right->Clone()), _op(b._op){

}
BinaryOperationNode::~BinaryOperationNode() {

    delete _left;
    delete _right;
}

Type* BinaryOperationNode::Execute(SymbolTable& st) const {

    Type* left = _left->Execute(st);
    Type* right = _right->Execute(st);

    SkupType* s_left = dynamic_cast<SkupType*>(left);
    if (s_left == nullptr && _op != "in")
        throw std::invalid_argument("BinaryOperationNode: Levi argument mora biti skup");
    else if (_op == "in") {
        IntType* broj = dynamic_cast<IntType*>(left);
        if (broj == nullptr)
            throw std::invalid_argument("BinaryOperationNode: Levi argument mora biti broj");
        s_left = new SkupType(broj->GetInt());
        delete broj;
        left = s_left;
    }

    SkupType* s_right = dynamic_cast<SkupType*>(right);
    if (s_right == nullptr)
        throw std::invalid_argument("BinaryOperationNode: Desni argument mora biti skup");

    Type* result = nullptr;
    if (_op == "+")
        result = new SkupType(s_left->GetSkup() + s_right->GetSkup());
    else if (_op == "*")
        result = new SkupType(s_left->GetSkup() * s_right->GetSkup());
    else if (_op == "-")
        result = new SkupType(s_left->GetSkup() - s_right->GetSkup());
    else if (_op == "==")
        result = new BoolType(s_left->GetSkup() == s_right->GetSkup());
    else if (_op == "!=")
        result = new BoolType(s_left->GetSkup() == s_right->GetSkup());
    else if (_op == "in")
        result = new BoolType(s_right->GetSkup().Sadrzi(s_left->GetSkup()));
    else if (_op == "sub")
        result = new BoolType(s_right->GetSkup().Sadrzi(s_left->GetSkup()));
    else
        throw std::invalid_argument("BinaryOperationNode: Nepoznata operacija");

    delete left;
    delete right;

    return result;
}
void BinaryOperationNode::Compile(SymbolTable& st, std::ostream& s) const {
    
    _left->Compile(st, s);
    s << " " <<  _op << " ";
    _right->Compile(st, s);
}
SynTreeNode* BinaryOperationNode::Clone() const {
    return new BinaryOperationNode(*this);
}
void BinaryOperationNode::show(std::ostream& s) const {

    _left->show(s);
    s << " " <<  _op << " ";
    _right->show(s);
}

IdentifierNode::IdentifierNode(const std::string& id) 
    : _id(id) {

}

Type* IdentifierNode::Execute(SymbolTable& st) const {

    Type* var = st.GetVar(_id);
    if (var == nullptr)
        throw std::invalid_argument("IdentifierNode: Promenljiva nije definisana");

    return var->Clone();
}
void IdentifierNode::Compile(SymbolTable& st, std::ostream& s) const {
    s << _id;
}
SynTreeNode* IdentifierNode::Clone() const {
    return new IdentifierNode(*this);
}
void IdentifierNode::show(std::ostream& s) const {
    s << _id;
}

ComplementNode::ComplementNode(SynTreeNode* exp)
    : _exp(exp){

}
ComplementNode::ComplementNode(const ComplementNode& cn)
    : _exp(cn._exp->Clone()){

}
ComplementNode::~ComplementNode(){
    delete _exp;
}

Type* ComplementNode::Execute(SymbolTable& st) const{

    Type* result = _exp->Execute(st);
    SkupType* s_type = dynamic_cast<SkupType*>(result);
    if (s_type == nullptr)
        throw std::invalid_argument("ComplementNode: Argument mora biti skup");

    Skup s = s_type->GetSkup();
    delete result;

    return new SkupType(univerzalni - s);
}
void ComplementNode::Compile(SymbolTable& st, std::ostream& s) const{
    s << "~";
    _exp->Compile(st, s);
}
SynTreeNode* ComplementNode::Clone() const{
    return new ComplementNode(*this);
}
void ComplementNode::show(std::ostream& s) const{

    s << "~";
    _exp->show(s);
}


UniverzalniNode::UniverzalniNode(SynTreeNode* left, SynTreeNode* right) 
    : _left(left), _right(right) {

}
UniverzalniNode::UniverzalniNode(const UniverzalniNode& un) 
    : _left(nullptr), _right(nullptr) {

    if (un._left != nullptr)
        _left = un._left->Clone();
    if (un._right != nullptr)
        _right = un._right->Clone();
}
UniverzalniNode::~UniverzalniNode() {
    if (_left != nullptr)
        delete _left;
    if (_right != nullptr)
        delete _right;
}

Type* UniverzalniNode::Execute(SymbolTable& st) const {

    if (_left == nullptr && _right == nullptr) {
        return new SkupType(univerzalni);
    }
    else {
        Type* left = _left->Execute(st);
        Type* right = _right->Execute(st);
        
        IntType* l = dynamic_cast<IntType*>(left);
        if (l == nullptr)
            throw std::invalid_argument("UniverzalniNode: Levi argument mora biti broj");
        IntType* r = dynamic_cast<IntType*>(right);
        if (r == nullptr)
            throw std::invalid_argument("UniverzalniNode: Desni argument mora biti broj");

        univerzalni = Skup(l->GetInt(), r->GetInt());

        delete left;
        delete right;

        return nullptr;
    }
}
void UniverzalniNode::Compile(SymbolTable& st, std::ostream& s) const {

    if (_left == nullptr && _right == nullptr) {
        s << "univ";
    }
    else {
        s << "univ = [";
        _left->Compile(st, s);
        s << ", ";
        _right->Compile(st, s);
        s << "]";
    }
}
SynTreeNode* UniverzalniNode::Clone() const {
    return new UniverzalniNode(*this);
}
void UniverzalniNode::show(std::ostream& s) const {
    if (_left == nullptr && _right == nullptr) {
        s << "univ";
    }
    else {
        s << "univ = [";
        _left->show(s);
        s << ", ";
        _right->show(s);
        s << "]";
    }
}

LogickiNode::LogickiNode(SynTreeNode* exp)
    : _exp(exp) {

}
LogickiNode::LogickiNode(const LogickiNode& ln)
    : _exp(ln._exp->Clone()){

}
LogickiNode::~LogickiNode(){
    delete _exp;
}

Type* LogickiNode::Execute(SymbolTable& st) const{

    Type* result = _exp->Execute(st);
    BoolType* b = dynamic_cast<BoolType*>(result);

    if (b == nullptr)
        throw std::invalid_argument("LogickiNode: Rezultat mora biti bool");

    std::cout << (b->GetBool() ? "True" : "False") << std::endl;

    delete result;

    return nullptr;
}
void LogickiNode::Compile(SymbolTable& st, std::ostream& s) const{

    _exp->Compile(st, s);
    s << ";";
}
SynTreeNode* LogickiNode::Clone() const{

    return new LogickiNode(*this);
}
void LogickiNode::show(std::ostream& s) const{

    _exp->show(s);
    s << ";";
}

PrintNode::PrintNode(SynTreeNode* exp)
    : _exp(exp){

}
PrintNode::PrintNode(const PrintNode& ln)
    : _exp(ln._exp->Clone()){

}
PrintNode::~PrintNode(){
    delete _exp;
}

Type* PrintNode::Execute(SymbolTable& st) const{

    Type* result = _exp->Execute(st);

    std::cout << (result) << std::endl;

    delete result;
    return nullptr;
}
void PrintNode::Compile(SymbolTable& st, std::ostream& s) const{

    s << "print(";
    _exp->Compile(st, s);
    s << ");";
}
SynTreeNode* PrintNode::Clone() const{
    return new PrintNode(*this);
}
void PrintNode::show(std::ostream& s) const{

    s << "print(";
    _exp->show(s);
    s << ");";
}

DefinitionNode::DefinitionNode(const std::string& id, SynTreeNode* exp, bool update)
    : _id(id), _exp(exp), _update(update){

}
DefinitionNode::DefinitionNode(const DefinitionNode& ln)
    : _id(ln._id), _exp(nullptr), _update(ln._update) {
    
    if (ln._exp != nullptr)
        _exp = ln._exp->Clone();
}
DefinitionNode::~DefinitionNode(){
    if (_exp != nullptr)
        delete _exp;
}

Type* DefinitionNode::Execute(SymbolTable& st) const {

    if (_update) {
        if (_exp == nullptr)
            throw std::invalid_argument("DefinitionNode: Argument ne sme biti nullptr");

        Type* result = _exp->Execute(st);
        SkupType* skup = dynamic_cast<SkupType*>(result);
        if (skup == nullptr)
            throw std::invalid_argument("DefinitionNode: Argument mora biti skup");

        if (st.Update(_id, result) == false)
            throw std::invalid_argument("DefinitionNode: Promenljiva mora biti definisana");
    }
    else {

        Type* result = nullptr;
        if (_exp != nullptr) {
            result = _exp->Execute(st);
            SkupType* skup = dynamic_cast<SkupType*>(result);
            if (skup == nullptr)
                throw std::invalid_argument("DefinitionNode: Argument mora biti skup");
        }
        else {
            result = new SkupType(Skup());
        }

        if (st.Define(_id, result) == false)
            throw std::invalid_argument("DefinitionNode: Promenljiva vec definisana");
    }

    return nullptr;
}
void DefinitionNode::Compile(SymbolTable& st, std::ostream& s) const {

    if (_update) {
        s << "def " << _id;
        if (_exp != nullptr) {
            s << " = ";
            _exp->Compile(st, s);
        }
        s << ";";
    }
    else {
        s << _id << " = ";
        if (_exp == nullptr)
            throw std::invalid_argument("DefinitionNode: Argument ne sme biti null");
        _exp->Compile(st, s);
        s << ";";
    }
}
SynTreeNode* DefinitionNode::Clone() const {

    return new DefinitionNode(*this);
}
void DefinitionNode::show(std::ostream& s) const {
    if (_update) {
        s << _id << " = ";
        if (_exp == nullptr)
            throw std::invalid_argument("DefinitionNode: Argument ne sme biti null");
        _exp->show(s);
        s << ";";
    }
    else {
        s << "def " << _id;
        if (_exp != nullptr) {
            s << " = ";
            _exp->show(s);
        }
        s << ";";
    }
}

SequenceNode::SequenceNode(SynTreeNode* exp)
    : _stmts() {

        _stmts.push_back(exp);
}
SequenceNode::SequenceNode(const SequenceNode& ln) 
    : _stmts(){

    for (auto& v : ln._stmts)
        _stmts.push_back(v->Clone());
}
SequenceNode::~SequenceNode() {

    for (auto& v : _stmts)
        delete v;
}

Type* SequenceNode::Execute(SymbolTable& st) const {

    for (auto& v : _stmts) {
        v->Execute(st);
    }
    return nullptr;
}
void SequenceNode::Compile(SymbolTable& st, std::ostream& s) const {

    for (auto& v : _stmts) {
        v->Compile(st, s);
        s << std::endl;
    }
}
SynTreeNode* SequenceNode::Clone() const {
    return new SequenceNode(*this);
}
void SequenceNode::show(std::ostream& s) const {

    for (auto& v : _stmts) {
        v->show(s);
        s << std::endl;
    }
}

void SequenceNode::Dodaj(SynTreeNode* exp) {

    _stmts.push_back(exp);
}